
/*
  This file implements CInt3List class,
  which is used to manupulate with INT3 (0xCC) bytes
  within debugging process.
*/

#define LOG_INT3

#define INT3_STATUS_DISABLED    0
#define INT3_STATUS_ENABLED     1

struct int3_struct : CListEntry
{
  DWORD addr;          // address in the context of the tracing process
  BYTE  origbyte;      // original byte
  BYTE  status;        // INT3_STATUS_xxx
};

class CInt3List : public CList
{
  public:
  CInt3List() : CList( sizeof(int3_struct) ) {}
  int Insert (HANDLE handle, DWORD addr);        // returns: 1 = OK
  int Disable(HANDLE handle, DWORD addr);        //          0 = error
  int DeleteAll(HANDLE handle);
}; // CInt3List

int CInt3List::Insert(HANDLE handle, DWORD addr)
{

  BYTE origbyte;
  if (read_memory(handle, addr, &origbyte, 1) == 0)
  {
#ifdef LOG_INT3
    log("ERROR: CInt3List::Insert(addr=%08X): cant read origbyte\n", addr);
#endif
    return 0;
  }

  int3_struct* t;
  ForEach(int3_struct, t)
  if (t->addr == addr)
  {
//#ifdef LOG_INT3
//    log("WARNING: CInt3List::Insert(addr=%08X): INT3 alredy assigned to this addr\n", addr);
//#endif
    if (t->status == INT3_STATUS_ENABLED)
    {
//#ifdef LOG_INT3
//      log("WARNING: CInt3List::Insert(addr=%08X): INT3 alredy enabled at this addr\n", addr);
//#endif
      if (origbyte != 0xCC)
      {
#ifdef LOG_INT3
        log("ERROR: CInt3List::Insert(addr=%08X): INT3 enabled, but CC-byte not found at this addr, now=%02X\n", addr, origbyte);
#endif
        return 0;
      }
      return 1;
    }
    else
    { // t->status == INT3_STATUS_DISABLED
//#ifdef LOG_INT3
//      log("WARNING: CInt3List::Insert(addr=%08X): enabling INT3\n", addr);
//#endif
      if (origbyte != t->origbyte)
      {
#ifdef LOG_INT3
        log("ERROR: CInt3List::Insert(addr=%08X): INT3 disabled, but origbyte doesnt match, origbyte=%02X, now=%02X\n", addr, t->origbyte, origbyte);
#endif
        return 0;
      }
      BYTE cc = 0xCC;
      if (write_memory(handle, addr, &cc, 1) == 0)
      {
#ifdef LOG_INT3
        log("ERROR: CInt3List::Insert(addr=%08X): cant write CC byte\n", addr);
#endif
        return 0;
      }
      t->status = INT3_STATUS_ENABLED;
      return 1;
    }
  }

  // not found, add new entry

  BYTE cc = 0xCC;
  if (write_memory(handle, addr, &cc, 1) == 0)
  {
#ifdef LOG_INT3
    log("ERROR: CInt3List::Insert(addr=%08X): cant write CC byte\n", addr);
#endif
    return 0;
  }

  t = (int3_struct*) Alloc();
  if (t == NULL)
  {
#ifdef LOG_INT3
    log("ERROR: CInt3List::Insert(addr=%08X): cant allocate memory for int3_struct\n", addr);
#endif
    return 0;
  }

  t->addr     = addr;
  t->origbyte = origbyte;
  t->status   = INT3_STATUS_ENABLED;

  Attach(t);

  return 1;
} // CInt3List::Insert()

int CInt3List::Disable(HANDLE handle, DWORD addr)
{

  BYTE cc;
  if (read_memory(handle, addr, &cc, 1) == 0)
  {
#ifdef LOG_INT3
    log("ERROR: CInt3List::Disable(addr=%08X): cant read byte\n", addr);
#endif
    return 0;
  }

  int3_struct* t;
  ForEach(int3_struct, t)
  if (t->addr == addr)
  {
    if (t->status == INT3_STATUS_ENABLED)
    {
      if (cc != 0xCC)
      {
#ifdef LOG_INT3
        log("ERROR: CInt3List::Disable(addr=%08X): byte has been changed from CC, now = %02X, origbyte = %02X\n", addr, cc, t->origbyte);
#endif
        return 0;
      }
      if (write_memory(handle, addr, &t->origbyte, 1) == 0)
      {
#ifdef LOG_INT3
        log("ERROR: CInt3List::Disable(addr=%08X): cant write origbyte\n", addr);
#endif
        return 0;
      }
      t->status = INT3_STATUS_DISABLED;
      return 1;
    }
    else
    { // t->status == INT3_STATUS_DISABLED
      if (cc != t->origbyte)
      {
#ifdef LOG_INT3
        log("ERROR: CInt3List::Disable(addr=%08X): INT3 disabled, but origbyte doesnt match, origbyte=%02X, now=%02X\n", addr, t->origbyte, cc);
#endif
        return 0;
      }
      return 1;
    }
  }
#ifdef LOG_INT3
  log("ERROR: CInt3List::Disable(addr=%08X): addr not found in list\n", addr);
#endif
  return 0;
} // CInt3List::Disable()

int CInt3List::DeleteAll(HANDLE handle)
{
  int3_struct* t;
  ForEach(int3_struct, t)
  {

    BYTE cc;
    if (read_memory(handle, t->addr, &cc, 1) == 0)
    {
  #ifdef LOG_INT3
      log("ERROR: CInt3List::DeleteAll(addr=%08X): cant read byte\n", t->addr);
  #endif
      return 0;
    }

    if (t->status == INT3_STATUS_ENABLED)
    {
      if (cc != 0xCC)
      {
#ifdef LOG_INT3
        log("ERROR: CInt3List::DeleteAll(addr=%08X): byte has been changed from CC, now = %02X, origbyte = %02X\n", t->addr, cc, t->origbyte);
#endif
        return 0;
      }
      if (write_memory(handle, t->addr, &t->origbyte, 1) == 0)
      {
#ifdef LOG_INT3
        log("ERROR: CInt3List::DeleteAll(addr=%08X): cant write origbyte\n", t->addr);
#endif
        return 0;
      }
//    t->status = INT3_STATUS_DISABLED;
    }
    else
    {
      if (cc != t->origbyte)
      {
#ifdef LOG_INT3
        log("ERROR: CInt3List::DeleteAll(addr=%08X): INT3 disabled, but origbyte doesnt match, origbyte=%02X, now=%02X\n", t->addr, t->origbyte, cc);
#endif
        return 0;
      }
    }

  } // ForEach

  return 1;
} // CInt3List::DeleteAll()
